'use strict';

describe('services/studyApproval.js', function () {

  var $studyApproval, $q, $scope, $rootScope, $api = null;

  beforeEach(module('genisis'));

  beforeEach(function () {
    inject(function (studyApproval, _$q_, _$rootScope_, $httpBackend,
      api) {
      $studyApproval = studyApproval;
      $q = _$q_;
      $api = api;
      $rootScope = _$rootScope_;
      $rootScope.errors = [];
      $scope = $rootScope.$new();

      $httpBackend.when('GET').respond('test GET response');
    });
  });

  it('service exists', function () {
    expect($studyApproval).not.toBeNull();
  });

  describe('$studyApproval.toName() function', function () {
    it('exists', function () {
      expect($studyApproval.toName).toBeDefined();
    });

    it('returns None if it cannot find a name for the study approval',
      function () {
        expect($studyApproval.toName(999)).toEqual('None');
      }
    );

    it('returns None if no study ID is given', function () {
      expect($studyApproval.toName()).toEqual('None');
    });

    it('returns a name for a study approval', function () {

      var fakeResponse = {
        id: 1,
        title: 'testing'
      };

      spyOn($api, 'fetch').and.callFake(function () {
        var deferred = $q.defer();
        deferred.resolve({
          success: true,
          response: fakeResponse
        });
        return deferred.promise;
      });

      $studyApproval.load();
      $scope.$apply();
      expect($studyApproval.toName(1)).toEqual(fakeResponse.title);
    });
  });

  describe('$studyApproval.load() function', function () {
    it('exists', function () {
      expect($studyApproval.load).toBeDefined();
    });

    it('loads data', function () {
      var fakeResponse = [{
        id: 1,
        title: 'testing'
      }, {
        id: 2,
        title: 'another test'
      }];

      spyOn($api, 'fetch').and.callFake(function () {
        var deferred = $q.defer();
        deferred.resolve({
          success: true,
          response: fakeResponse
        });
        return deferred.promise;
      });

      var promise = $studyApproval.load();

      promise.then(function (data) {
        expect(data).toEqual(fakeResponse);
      });

      $scope.$apply();
    });

    it('returns an array if only one record is found', function () {
      var fakeResponse = {
        id: 1,
        title: 'testing'
      };

      spyOn($api, 'fetch').and.callFake(function () {
        var deferred = $q.defer();
        deferred.resolve({
          success: true,
          response: fakeResponse
        });
        return deferred.promise;
      });

      var promise = $studyApproval.load();

      promise.then(function (data) {
        expect(data).toEqual([fakeResponse]);
      });

      $scope.$apply();
    });

    it('returns an empty array if there is no response', function () {

      spyOn($api, 'fetch').and.callFake(function () {
        var deferred = $q.defer();
        deferred.resolve({
          success: false
        });
        return deferred.promise;
      });

      var promise = $studyApproval.load();

      promise.then(function (data) {
        expect(data).toEqual([]);
      });

      $scope.$apply();
    });
  });

  describe('$studyApproval.find() function', function () {
    it('exists', function () {
      expect($studyApproval.find).toBeDefined();
    });

    it('returns null if no ID is given', function () {
      expect($studyApproval.find()).toEqual(null);
    });

    it('returns null if the ID does not exist', function () {
      expect($studyApproval.find(999)).toEqual(null);
    });

    it('returns the study approval for the given ID', function () {
      var fakeResponse = {
        id: 1,
        title: 'testing'
      };

      spyOn($api, 'fetch').and.callFake(function () {
        var deferred = $q.defer();
        deferred.resolve({
          success: true,
          response: fakeResponse
        });
        return deferred.promise;
      });

      $studyApproval.load();
      $scope.$apply();

      expect($studyApproval.find(1)).toEqual(fakeResponse);
    });
  });

  describe('$studyApproval.studies() function', function () {
    it('exists', function () {
      expect($studyApproval.studies).toBeDefined();
    });

    it('returns an empy array if there are no studies loaded', function () {
      expect($studyApproval.studies()).toEqual([]);
    });

    it('returns an array of studies', function () {
      var fakeResponse = [{
        id: 1,
        title: 'testing'
      }, {
        id: 2,
        title: 'another test'
      }];

      spyOn($api, 'fetch').and.callFake(function () {
        var deferred = $q.defer();
        deferred.resolve({
          success: true,
          response: fakeResponse
        });
        return deferred.promise;
      });

      $studyApproval.load();
      $scope.$apply();
      expect($studyApproval.studies()).toEqual(fakeResponse);
    });
  });

  describe('$studyApproval.createApproval() function', function () {
    it('exists', function () {
      expect($studyApproval.createApproval).toBeDefined();
    });

    it('creates a study approval', function () {

      var fakeStudy = {
        id: 2,
        title: 'another test'
      };
      var fakeResponse = {
        response: fakeStudy,
        success: true
      };

      spyOn($api, 'create').and.callFake(function () {
        var deferred = $q.defer();
        deferred.resolve(fakeResponse);
        return deferred.promise;
      });

      var promise = $studyApproval.createApproval(fakeStudy);
      promise.then(function (data) {
        expect(data).toEqual(2);
        expect($studyApproval.studies()).toEqual([fakeStudy]);
      });
      $scope.$apply();
    });

    it('returns false if there is an invalid response', function () {

      var fakeResponse = {
        response: null,
        success: false
      };

      spyOn($api, 'create').and.callFake(function () {
        var deferred = $q.defer();
        deferred.resolve(fakeResponse);
        return deferred.promise;
      });

      var promise = $studyApproval.createApproval();

      promise.then(function (data) {
        expect(data).toEqual(false);
      });

      $scope.$apply();
      expect($studyApproval.studies()).toEqual([]);
    });
  });

  describe('$studyApproval.valid() function', function () {
    it('exists', function () {
      expect($studyApproval.valid).toBeDefined();
    });

    it('returns $rootScope.errors if there is no title', function () {
      var fakeApproval = {};

      expect($studyApproval.valid(fakeApproval)).toEqual(false);
      expect($rootScope.errors).toEqual(
        ['You must enter a study approval title.']
      );
    });

    it('returns $rootScope.errors if there is no obtained', function () {
      var fakeApproval = {
        title: 'test'
      };

      expect($studyApproval.valid(fakeApproval)).toEqual(false);
      expect($rootScope.errors).toEqual(
        ['You must enter where the study approval was obtained.']
      );
    });

    it('returns $rootScope.errors if there is no description', function () {
      var fakeApproval = {
        title: 'test',
        obtained: 'lala land'
      };

      expect($studyApproval.valid(fakeApproval)).toEqual(false);
      expect($rootScope.errors).toEqual(
        ['You must enter a description of the study approval.']
      );
    });

    it('returns true if there are no errors', function () {
      var fakeApproval = {
        title: 'test',
        obtained: 'lala land',
        description: 'testing 123'
      };
      expect($studyApproval.valid(fakeApproval)).toEqual(true);
      expect($rootScope.errors).toEqual([]);
    });
  });
});